# +------------------------------------------------------------------------+
# | Einfuehrung in die Oekonometrie                                        |
# | Programmierung mit R                                                   |
# | Beispiele "Konzentrationsrechnung", "Intervallschaetzung"              |
# | R. Schuhr 15.03.2012                                                   |
# + -----------------------------------------------------------------------+


# Beispiel 1: Konzentrationsrechnung (doc_konzentrationsrechnung.pdf)
# -------------------------------------------------------------------
#
nj <- c(13, 1, 1, 1, 1, 1, 1, 1) # Vektor der absoluten Klassenhaeufigkeiten
sj <- c(25.0, 6.0, 7.0, 10.0, 12.0, 13.0, 13.5, 13.5) # Vektor der Klassenmerkmalssummen 
#
sum(nj)                          # Anzahl der TV-Sender
hj <- nj/sum(nj)                 # relative Klassenhaeufigkeiten
hj
Hj <- cumsum(hj)                 # kumulierte relative Klassenhaeufigkeiten
Hj
#
sum(sj)                          # Merkmalssumme
qj <- sj/sum(sj)                 # Klassenanteile an der Merkmalssumme 
qj
Qj <- cumsum(qj)                 # kumulierte Klassenanteile an der Merkmalssumme
Qj
#
tabelle <- cbind(hj,Hj,qj,Qj)    # Tabelle
tabelle
#
Hj <- c(0,Hj)                    # Lorenzkurve
Qj <- c(0,Qj)
plot(Hj, Qj, type="b", , col="blue", xlab="relative Klassenhaeufigkeit", 
     ylab="Marktanteil", main="Lorenzkurve")                     
lines(Hj, Hj)

# Verwendete R-Funktionen
?c
?cbind
?sum
?cumsum
?plot
?lines

QQj  <- Qj[1:8]+Qj[2:9]          # Berechnung des Gini-Koeffizienten
hQQj <- QQj*hj
tabelle <- cbind(Hj,Qj,c(0,QQj),c(0,hQQj))
tabelle                          # Berechnungstabelle
round(tabelle, digits=3)         # formatierte Berechnungstabelle
sum(hQQj)

F   <- 0.5 - (0.5*sum(hQQj))     # Konzentrationsflaeche
GK  <- 2*F                       # Gini-Koeffizient
GKK <- GK * 20/19                # normierter Gini-Koeffizient
round(c(F,GK,GKK),digits=3)

# Verwendete R-Funktion
?round


# Eigene Programmierung von R-Funktionen 
# Lorenzkurve
lorenz <- function(nj, sj) {
          m  <- length(nj)
          m1 <- m+1
          nj <- c(0,nj)
          sj <- c(0,sj)
          hj <- nj/sum(nj)
          Hj <- cumsum(hj)
          qj <- sj/sum(sj)
          Qj <- cumsum(qj)
          plot(Hj, Qj, type="b", , col="blue", main="Lorenzkurve")  
          lines(Hj,Hj)
          lorenz <- cbind(hj,qj,Hj,Qj)
          lorenz <- lorenz[2:m1,]
          lorenz
          }

# Gini-Koeffizient
gini   <- function(nj, sj) {
          m1 <- length(nj)-1
          nn <- sum(nj)
          L  <- lorenz(nj,sj) 
          QQj  <- c(0,L[1:m1,4])+L[,4]
          hQQj <- QQj*L[,1]
          GK   <- 2*(0.5 - (0.5*sum(hQQj)))
          gini <- GK * nn/(nn-1)
          gini
          }

# Anwendung
lorenz(nj,sj)  			    # Lorenz-Kurve
tab <- lorenz(nj,sj)              
round(tab,digits=2)               # Formatierte Ausgabe der Ergebnisse
gini(nj,sj)                       # Gini-Koeffizient



# Beispiel 2: Intervallschaetzung (doc_intervallschaetzung.pdf)
# -------------------------------------------------------------
#
# Stichprobe
feinstaub <- c(50.8, 45.9, 46.4, 56.1, 50.1 ,44.9, 42.3, 45.2, 47.6, 53.6, 
               45.6, 44.6, 41.3, 44.4, 41.0, 47.6, 54.0, 50.4, 45.5, 44.2, 
               48.0, 45.7, 50.1, 49.1, 52.9, 41.8, 42.4, 47.9, 48.8, 41.9)

# Stichprobenmomente
n  <- length(feinstaub)                # Stichprobenumfang
n
smw <- sum(feinstaub)/n                # Stichprobenmittelwert
smw
svr <- sum( (feinstaub-smw)**2 )/(n-1) # Stichprobenvarianz
svr
sqrt(svr)                              # Stichprobenstandardabweichung

# die Berechnungen etwas krzer...
mean(feinstaub)                        # Stichprobenmittelwert
var(feinstaub)                         # Stichprobenvarianz
sd(feinstaub)                          # Stichprobenstandardabweichung

# Konfidenzintervall fr den Mittelwert mu einer normalverteilten Grundgesamtheit
sfmw <- sqrt( var(feinstaub)/n)        # Standardfehler des Stichprobenmittelwerts
sfmw 
ndf <- n-1                             # Anzahl der Freiheitsgrade
alpha <- 0.01                          # Irrtumswahrscheinlichkeit
t <- qt((1-alpha/2), ndf)              # (1-alpha/2)-Quantil der t(ndf)-Verteilung
t
kiu <- smw - t*sfmw                    # (1-alpha)-Konfidenzintervall fr mu
kio <- smw + t*sfmw
c(kiu, kio)
round(c(kiu, kio), digits=3)

# Verwendete R-Funktion
?qt  # siehe auch "dt", "pt", "rt" sowie "qnorm", "dnorm", "pnorm", "rnorm", "qchisq", ...

# Exkurs: Anwendung der R-Funktionen "dt", "pt", "rt"
x <- seq(-4,4,0.05)       
x
dt(x,29)                               
plot(x, dt(x,29), type="l", xlab="x", ylab="f(x)", 
     main="Dichte der t(29)-Verteilung")
pt(x,29)
plot(x, pt(x,29), type="l", xlab="x", ylab="f(x)", 
     main="Verteilungsfunktion der t(29)-Verteilung")
rt(10,29)
hist(rt(500,29), breaks=12, xlab="ZZ", ylab="Haeufigkeit", 
     main="500 Zufallszahlen aus einer t(29)-Verteilung")


# Eigene Programmierung von R-Funktionen 
# Konfidenzintervall fr den Mittelwert mu einer normalverteilten Grundgesamtheit
konfintmu <- function(x, alpha=0.05) {
             n <- length(x)
             mw<-mean(x) ; sfmw<-sqrt( var(x)/n) 
             t <- qt(1-alpha/2, n-1)
             konfintmu <- c(mw-t*sfmw,mw+t*sfmw)
             konfintmu 
             }


# Anwendung
mean(feinstaub)              # Stichprobenmittel
konfintmu(feinstaub)         # 95%-Konfidenzintervall fr mu
konfintmu(feinstaub,0.01)    # 99%-Konfidenzintervall fr mu
konfintmu(feinstaub,0.001)   # 99.9%-Konfidenzintervall fr mu

feinstaub
feinstaub>50                 # Tage mit mehr als 50 ug Feinstaub in der Luft
mean(feinstaub>50)           # Anteil der Tage mit mehr als 50 ug Feinstaub in der Luft
konfintmu(feinstaub>50)      # approx. 95%-Konfidenzintervall fr den Anteilssatz

365*mean(feinstaub>50)       # 'Hochrechnung' auf ein Jahr
365*konfintmu(feinstaub>50)



# Beispiel 3: Intervallschaetzung mittels Bootstrapping: Perzentil-Methode (Quantil-Methode)
# ------------------------------------------------------------------------------------------
# Siehe z.B. R. Schlittgen: Statistische Auswertungen mit R, S. 91ff

bskonfintmu <- function(x, bw=1000, alpha=0.05, smooth=1.5) {
               xmit <- mean(x)
               n <- length(x)
               m <- rep(0,bw)
               for (b in 1:bw) {
                    bs <- sample(x,n,replace=TRUE)
                    m[b]<- mean(bs)
                    }
               cf <- xmit+quantile((xmit-m), c((0.5*alpha),(1-0.5*alpha)))
               hist(m, freq=FALSE, xlab="Mittelwert", ylab="Dichte", main="") 
               lines(density(m, kernel="gaussian", adjust=smooth), col="blue")
               title(main=list("Bootstrapverteilung des arithmetischen Mittels", cex=0.9))
               print("Bootstrap-Konfidenzintervall") ; print(cf)
               bskonfintmu <- cf
               }                                                                                                                                                                                                                                                                                                                                                           
# Verwendete R-Funktion
?sample
?quantile


# Anwendung
mean(feinstaub)                     # Stichprobenmittel
bskonfintmu(feinstaub)              # 95%-Bootstrap-Konfidenzintervall fuer mu
bskonfintmu(feinstaub, 5000, 0.05)
konfintmu(feinstaub, 0.05)          # Vergleich mit "klassischem" Konfidenzintervall

mean(feinstaub>50)                  # Anteil der Tage mit mehr als 50 ug Feinstaub in der Luft
bskonfintmu(feinstaub>50, 10000, 0.05, 2)  # 95%-Bootstrap-Konfidenzintervall fuer den Anteilssatz
konfintmu(feinstaub>50, 0.05)       # Vergleich mit "klassischem" Konfidenzintervall

# Ende
